package brs.wurm.mod.server.milkfix;

import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gotti.wurmunlimited.modloader.interfaces.Configurable;
import org.gotti.wurmunlimited.modloader.interfaces.PlayerMessageListener;
import org.gotti.wurmunlimited.modloader.interfaces.WurmServerMod;

import com.wurmonline.server.creatures.Communicator;
import com.wurmonline.server.creatures.Creature;
import com.wurmonline.server.creatures.Creatures;

/** Corrects the bug in the Wurm Unlimited server code that never resets the milking and shearing flags.
 * <p>This is adapted from Cuddles' Milk Reset Mod, and is not recommended to be used with it.
 * @author Brian Sutton
 *
 */
public class MilkFix implements PlayerMessageListener, WurmServerMod, Configurable {
	
	
	/** Logging output to the console. */
	static Logger logger = Logger.getLogger(MilkFix.class.getName());
	
    /** For testing.
     * @param args
     * @throws InterruptedException
     */
    static public void main(String[] args) throws InterruptedException {
    	MilkFix instance = new MilkFix();
    	Properties props = new Properties();
    	props.setProperty("milkhours", "3");
    	props.setProperty("shearhours", "168");
    	instance.configure(props);
    	System.out.println(instance.getTimeToMilk());
    	instance.lastMilkableReset -= 60000;
    	System.out.println(instance.getTimeToMilk());
    	instance.lastMilkableReset -= 60000 * 65;
    	System.out.println(instance.getTimeToMilk());
    	System.out.println(instance.getTimeToShear());
    	Thread.sleep(2000);    	
    }
    
    /** The number of hours between milking being available. A value of zero disables the fix.*/
    private int milkHours;
    
    /** The number of hours between shearing being available. A value of zero disables the fix.*/
    private int shearHours;
    
    /** The delay between milking resets, in milliseconds. */
    private int milkingInterval; 
    
    /** The delay between shearing resets, in milliseconds. */
    private int shearingInterval; 
    
    /** The last system time milking was reset. Used to predict the next reset time.*/
    long lastMilkableReset = System.currentTimeMillis();
    
    /** The last system time shearing was reset. Used to predict the next reset time.*/
    long lastShearableReset = System.currentTimeMillis();
    
    /**
     * Starts the timer threads, if enabled.
     */
    private void startThreads() {
    	if (milkHours > 0) {
    		startMilkingThread();
    	}
    	if (shearHours > 0) {
    		startShearingThread();    	
    	}    	
    }
    
    /**
     * Starts the milking timer thread. This assumes milking is already enabled on startup.
     */
	private void startMilkingThread() {
		Thread milkingThread = new Thread() {
			public void run() {
				while (true) {
					try {
						Thread.sleep(milkingInterval);
						resetMilk();
					} catch (InterruptedException e) {
						logger.info("Milking thread interrupted.");
					}
				}
			}
		};
		milkingThread.setDaemon(true);
		milkingThread.start();
		logger.info("Milking timer started.");
    }
    
	/**
	 * Starts the shearing timer thread. This assumes shearing is enabled on startup.
	 */
	private void startShearingThread() {
		Thread shearingThread = new Thread() {
			public void run() {
				while (true) {
					try {
						Thread.sleep(shearingInterval);
						resetShear();
					} catch (InterruptedException e) {
						logger.info("Shearing thread interrupted.");
					}
				}
			}
		};
		shearingThread.setDaemon(true);
		shearingThread.start();
		logger.info("Shearing timer started.");
	}
    
    /**
     * Obtains this code's version.
     */
    @Override
    public String getVersion() {
    	return "v1.0";
    }
    
	/**
	 * Configures the properties and starts the timer threads.
	 */
	@Override
	public void configure(Properties properties) {
		String hours = properties.getProperty("milkhours", "0");
		try { // check for bad value
			milkHours = Math.max(0, Integer.parseInt(hours)); 
		} catch (NumberFormatException e) {
			logger.info("Invalid milkhours value: " + hours);
		}
		hours = properties.getProperty("shearhours", "0");
		try {
			shearHours = Math.max(0, Integer.parseInt(hours)); 
		} catch (NumberFormatException e) {
			logger.info("Invalid shearhours value: " + hours);			
		}
		milkingInterval = 1000 * 60 * 60 * milkHours;
		shearingInterval = 1000 * 60 * 60 * shearHours;
		StringBuilder info = new StringBuilder("MilkFix started, ");
		info.append(getVersion());
		info.append("\nHours between milking: ");
		info.append((milkHours == 0) ? "disabled" : milkHours);
		info.append("\nHours between shearing: ");
		info.append((shearHours == 0) ? "disabled" : shearHours);
		logger.info(info.toString());
		startThreads();
	}	


	/**
	 * Re-enables milking.
	 */
	public void resetMilk() {
		logger.log(Level.INFO, "Resetting milking.");
		Creature[] crets = Creatures.getInstance().getCreatures();
		for (int x = 0; x < crets.length; ++x) {
			if (crets[x].isMilkable()) {
				crets[x].setMilked(false);
			}
		}
		lastMilkableReset = System.currentTimeMillis();
	}
	
	/**
	 * Re-enables shearing.
	 */
	public void resetShear() {
		logger.log(Level.INFO, "Resetting shearing.");
		Creature[] crets = Creatures.getInstance().getCreatures();
		for (int x = 0; x < crets.length; ++x) {
			if (crets[x].isSheared()) {
			crets[x].setSheared(false);
			}
		}
		lastShearableReset = System.currentTimeMillis();
	}

	/** Adds the plural form, basically just adds an "s" if <i>value</i> does not equal 1.
	 * @param builder A StringBuilder.
	 * @param value The value to test.
	 */
	private void addPlural(StringBuilder builder, int value) {
		if (value != 1) {
			builder.append("s");
		}
	}
	
	/** Adds a time frame. Used when building a message string.
	 * @param builder A StringBuilder.
	 * @param type The name of the time period, i.e. day, hour, minute.
	 * @param value The value for the given time period.
	 */
	private void addTimeFrame(StringBuilder builder, String type, int value) {
		builder.append(value);
		builder.append(" ");
		builder.append(type);
		addPlural(builder, value);		
	}
	
	/** Builds a message string showing the days, hours, and minutes before the next reset.
	 * @param type The function type, i.e. milking or shearing.
	 * @param lastReset When the last reset occurred.
	 * @param interval The length of the interval between resets.
	 * @return A user readable message indicating when the next reset will occur.
	 */
	private String getTimeToReset(String type, long lastReset, long interval) {
		if (interval == 0) {
			return type + " fix is disabled.";
		} else {
			long millis = interval - (System.currentTimeMillis() - lastReset);
			long totalMinutes = Math.floorDiv(millis, 60000);
			int days = (int)Math.floorDiv(totalMinutes, 1440);
			totalMinutes = (int)Math.floorMod(totalMinutes, 1440);
			int hours = (int)Math.floorDiv(totalMinutes, 60);
			int minutes = (int)Math.floorMod(totalMinutes, 60);
			StringBuilder bob = new StringBuilder("Time before next ");
			bob.append(type);
			bob.append(" reset: ");
			if (days > 0) {
				addTimeFrame(bob, "day", days);
				bob.append(", ");
			}
			if (hours > 0) {
				addTimeFrame(bob, "hour", hours);
				bob.append(", ");
			}
			addTimeFrame(bob, "minute", minutes);
			return bob.toString();
		}
	}
	
	/** Obtains a message indicating how long it will be before the next milking reset.
	 * @return A formatted user message.
	 */
	private String getTimeToMilk() {
		return getTimeToReset("milking", lastMilkableReset, milkingInterval);
	}
	
	/** Obtains a message indicating how long it will be before the next shearing reset.
	 * @return A formatted user message.
	 */
	private String getTimeToShear() {
		return getTimeToReset("shearing", lastShearableReset, shearingInterval);
	}

	/**
	 * Checks for the /milk or /shear command and returns the respective message.
	 */
	@Override
	public boolean onPlayerMessage(Communicator communicator, String msg) {
		msg = msg.toLowerCase();
		if (msg.startsWith("/milk")) {
			communicator.sendNormalServerMessage(getTimeToMilk());
		} else if (msg.startsWith("/shear")) {
			communicator.sendNormalServerMessage(getTimeToShear());
		} else {
			return false;
		}
		return true;
	}
}