package brs.wurm.mod.server.lifex;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gotti.wurmunlimited.modloader.callbacks.CallbackApi;
import org.gotti.wurmunlimited.modloader.classhooks.HookException;
import org.gotti.wurmunlimited.modloader.classhooks.HookManager;
import org.gotti.wurmunlimited.modloader.interfaces.Configurable;
import org.gotti.wurmunlimited.modloader.interfaces.Initable;
import org.gotti.wurmunlimited.modloader.interfaces.PreInitable;
import org.gotti.wurmunlimited.modloader.interfaces.WurmServerMod;
import org.gotti.wurmunlimited.modsupport.creatures.CreatureTemplateParser;

import com.wurmonline.server.WurmCalendar;
import com.wurmonline.server.creatures.CreatureStatus;
import com.wurmonline.server.creatures.CreatureTemplateIds;

import javassist.CannotCompileException;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.NotFoundException;
import javassist.expr.ExprEditor;
import javassist.expr.FieldAccess;

/** Adapted from org.gotti.wurmunlimited.mods.creatureagemod.CreatureAgeMod, which does the opposite.
 * @author Brian Sutton
 * Age specifier:
# 0 : disabled
# 1-2 : young
# 3-7 : adolescent
# 8-11 : mature
# 12-29 : aged
# 30-39 : old
# 40+ : venerable 
	(CreatureStatus)
    public final String getAgeString() {
        if (this.age < 3) {
            return "young";
        }
        if (this.age < 8) {
            return "adolescent";
        }
        if (this.age < 12) {
            return "mature";
        }
        if (this.age < 30) {
            return "aged";
        }
        if (this.age < 40) {
            return "old";
        }
        return "venerable";
    }


 */
public class CreatureLifeExtender implements WurmServerMod, Configurable, Initable, PreInitable {

	private static final long ORIG_CREATURE_POLL_TIMER = 2419200L;	// 28 Wurm days, the normal time between age steps.
	private static final long INTERVAL_TIMER = ORIG_CREATURE_POLL_TIMER * 8;	// 8 wurm days per real life days.
	private static final long REAL_DAYS = INTERVAL_TIMER / 691200; 	// Number of real-life days between age changes.

	private static int extensionAge = 8;
	// private static long extendedLifeTimer = 259200L; // 72 wurm hours, the default for foals, hell foals, and unicorn foals
	private static Set<Integer> excludedTemplates = new HashSet<>(Arrays.asList(
			CreatureTemplateIds.BOAR_FO_CID,
			CreatureTemplateIds.HYENA_LIBILA_CID,
			CreatureTemplateIds.WORG_CID,
			CreatureTemplateIds.GORILLA_MAGRANON_CID,
			CreatureTemplateIds.ZOMBIE_CID,
			CreatureTemplateIds.SKELETON_CID
			));

	private final Logger logger = Logger.getLogger(this.getClass().getName());
	
	/** For testing.
	 * @param args
	 */
	static public void main(String[] args) {
		new CreatureLifeExtender().configure(null);
	}
	
	/**
	 * Version 1.0: Initial version, hard-coded to start at age 8, extend aging to 28 real life days.
	 */
	@Override
	public String getVersion() {
		return "1.0";
	}

	@Override
	public void configure(Properties properties) {
		final CreatureTemplateParser parser = new CreatureTemplateParser() {
			protected int unparsable(String name) {
				logger.warning("Invalid template " + name);
				return -1;
			}
		};

		logger.log(Level.INFO, "Growth slows at age: " + extensionAge);
		logger.log(Level.INFO, "Real days between aging: " + REAL_DAYS);		
		logger.log(Level.INFO, "Excluded templates: " + parser.toString(excludedTemplates.stream().mapToInt(Integer::intValue).toArray()));
	}
	
	// Foals age factor: 72 hours, otherwise 28 days. (Wurm time)  (28 wurm days = 3.5 RL days)
	@CallbackApi
	public long getAdjustedLastPolledAge(CreatureStatus creatureStatus, boolean reborn) {

		int age = creatureStatus.age;
		int templateId = creatureStatus.getTemplate().getTemplateId();
		boolean creatureQualifies = (!reborn) 													// not a zombie?
				&& (age >= extensionAge) 														// check if the age	qualifies					
				&& !excludedTemplates.contains(templateId);										// does this creature qualify?
		if (creatureQualifies) {																					// check the age?
			long timeSinceLastPoll = WurmCalendar.currentTime - creatureStatus.lastPolledAge;
			if (timeSinceLastPoll < INTERVAL_TIMER) {   // have we not yet passed the threshold to trigger advancing age?				
				return WurmCalendar.currentTime - ORIG_CREATURE_POLL_TIMER + 10;								// Trick the poller into thinking the last poll was less than 28 Wurm days ago.
			} else {
				StringBuilder bob = new StringBuilder("Increasing age for " );
				bob.append(creatureStatus.getTemplate().getName());
				bob.append(" from ");
				bob.append(age);
				bob.append(", at ");
				bob.append((int)(creatureStatus.getPositionX()/4));
				bob.append(", ");
				bob.append((int)(creatureStatus.getPositionY()/4));
				bob.append("; prev=");
				bob.append(creatureStatus.lastPolledAge);
				bob.append(", time=");
				bob.append(timeSinceLastPoll);
				logger.log(Level.INFO, bob.toString());
			}
		}	
		return creatureStatus.lastPolledAge;  // For inapplicable animals or when time to age, just use the actual last polled time.
	
	}
	
	@Override
	public void preInit() {
		try {
			
			final CtClass ctCreatureStatus = HookManager.getInstance().getClassPool().get("com.wurmonline.server.creatures.CreatureStatus");
			HookManager.getInstance().addCallback(ctCreatureStatus, "creatureagemod", this);	// Incompatible with creatureAgeMod, so that callback is used to prevent collisions.
			
			CtMethod method = ctCreatureStatus.getMethod("pollAge", "(I)Z");
			method.instrument(new ExprEditor() {
				
				@Override
				public void edit(FieldAccess f) throws CannotCompileException {
					if ("lastPolledAge".equals(f.getFieldName())) {
						f.replace("$_ = creatureagemod.getAdjustedLastPolledAge(this, reborn);");
					}
				}
			});
		
		} catch (NotFoundException | CannotCompileException e ) {
			throw new HookException(e);
		}
	}

	@Override
	public void init() {
	}

}


/*  From the Wurm Server code:
 

    final boolean pollAge(int maxAge) {
        boolean rebornPoll = false;
        if (this.reborn 
        	&& this.mother == -10L 
        	&& WurmCalendar.currentTime - this.lastPolledAge > 604800L) { 	// 1 week
            rebornPoll = true;
        }
        boolean fasterGrowth = this.statusHolder.getTemplate().getTemplateId() == 65 // foal
        	|| this.statusHolder.getTemplate().getTemplateId() == 117 	// HH foal
        	|| this.statusHolder.getTemplate().getTemplateId() == 118; 	// Uni foal
        	
        if (WurmCalendar.currentTime - this.lastPolledAge > 
        	(Servers.localServer.PVPSERVER && this.age < 8 	
        		&& fasterGrowth ? 259200L : 2419200L) // 72 hours vs 28 days
        		|| this.isTraitBitSet(29) 
        		&& WurmCalendar.currentTime - this.lastPolledAge > 345600L 
        		|| rebornPoll) {
		            if ((this.statusHolder.isGhost() 
		            	|| this.statusHolder.isKingdomGuard() 
		            	|| this.statusHolder.isUnique()) 
		            	&& (!this.reborn || this.mother == -10L)) {
		                	this.age = Math.max(this.age, 11);
	            }
	            
	    int newAge = this.age + 1;
        boolean updated = false;
        
            if (!this.statusHolder.isCaredFor() 
            	&& !this.isTraitBitSet(29) 
            	&& (newAge >= maxAge || this.isTraitBitSet(13) 
            	&& newAge >= Math.max(1, maxAge - Server.rand.nextInt(maxAge / 2)))) {
                return true;
            }
            if (!rebornPoll) {
                if (!(newAge <= (this.isTraitBitSet(21) ? 75 : (this.isTraitBitSet(29) ? 36 : 50)) || this.statusHolder.isGhost() || this.statusHolder.isHuman() || this.statusHolder.isUnique() || this.statusHolder.isCaredFor())) {
                    return true;
                }
                if (!(newAge - 1 < 5 || this.reborn || this.getTemplate().getAdultMaleTemplateId() <= -1 && this.getTemplate().getAdultFemaleTemplateId() <= -1)) {
                    int newtemplateId = this.getTemplate().getAdultMaleTemplateId();
                    if (this.sex == 1 && this.getTemplate().getAdultFemaleTemplateId() > -1) {
                        newtemplateId = this.getTemplate().getAdultFemaleTemplateId();
                    }
                    if (newtemplateId != this.getTemplate().getTemplateId()) {
                        newAge = 1;
                        try {
                            this.updateAge(newAge);
                            updated = true;
                        }
                        catch (IOException iox) {
                            logger.log(Level.WARNING, iox.getMessage(), iox);
                        }
                        try {
                            CreatureTemplate newTemplate;
                            this.setChanged(true);
                            this.statusHolder.template = this.template = (newTemplate = CreatureTemplateFactory.getInstance().getTemplate(newtemplateId));
                            if (!this.statusHolder.isNpc()) {
                                if (this.statusHolder.getMother() == -10L || !this.statusHolder.isHorse() && !this.statusHolder.isUnicorn() && !this.reborn) {
                                    try {
                                        if (this.statusHolder.getName().endsWith("traitor")) {
                                            this.statusHolder.setName(this.template.getName() + " traitor");
                                        } else {
                                            this.statusHolder.setName(this.template.getName());
                                        }
                                    }
                                    catch (Exception ex) {
                                        logger.log(Level.WARNING, ex.getMessage(), ex);
                                    }
                                }
                                if (!this.reborn || this.mother == -10L) {
                                    try {
                                        this.statusHolder.skills.delete();
                                        this.statusHolder.skills.clone(newTemplate.getSkills().getSkills());
                                        this.statusHolder.skills.save();
                                    }
                                    catch (Exception ex) {
                                        logger.log(Level.WARNING, ex.getMessage(), ex);
                                    }
                                }
                            }
                            this.save();
                            this.statusHolder.refreshVisible();
                        }
                        catch (NoSuchCreatureTemplateException nsc) {
                            logger.log(Level.WARNING, this.statusHolder.getName() + ", " + this.statusHolder.getWurmId() + ": " + nsc.getMessage(), nsc);
                        }
                        catch (IOException iox) {
                            logger.log(Level.WARNING, this.statusHolder.getName() + ", " + this.statusHolder.getWurmId() + ": " + iox.getMessage(), iox);
                        }
                    }
                }
            }
            if (!updated) {
                try {
                    this.updateAge(newAge);
                    if (this.statusHolder.getHitched() != null && !this.statusHolder.getHitched().isAnySeatOccupied(false) && !this.statusHolder.isDomestic() && this.getBattleRatingTypeModifier() > 1.2f) {
                        Server.getInstance().broadCastMessage(this.statusHolder.getName() + " stops dragging a " + Vehicle.getVehicleName(this.statusHolder.getHitched()) + ".", this.statusHolder.getTileX(), this.statusHolder.getTileY(), this.statusHolder.isOnSurface(), 5);
                        if (this.statusHolder.getHitched().removeDragger(this.statusHolder)) {
                            this.statusHolder.setHitched(null, false);
                        }
                    }
                }
                catch (IOException iox) {
                    logger.log(Level.WARNING, iox.getMessage(), iox);
                }
            }
        }
        return false;
    }

}

*/